@extends('backend.layouts.app')

@section('content')
    <div class="container-fluid">
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h4 class="mb-0">Create Manual Order</h4>
                    </div>
                    <div class="card-body">
                        <form action="{{ route('admin.orders.store') }}" method="POST" id="manual-order-form">
                            @csrf

                            <!-- Customer Information -->
                            <div class="row mb-4">
                                <div class="col-md-12">
                                    <h5 class="border-bottom pb-2 mb-3">Customer Information</h5>
                                </div>

                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label for="customer_id">Customer <span class="text-danger">*</span></label>
                                        <input name="customer_name" id="customer_id" class="form-control" required />


                                    </div>
                                </div>

                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label for="customer_phone">Phone Number <span class="text-danger">*</span></label>
                                        <input type="text" name="customer_phone" id="customer_phone" class="form-control" placeholder="Enter phone number" required>
                                    </div>
                                </div>
                            </div>

                            <!-- Shipping Address -->
                            <div class="row mb-4">
                                <div class="col-md-12">
                                    <h5 class="border-bottom pb-2 mb-3">Shipping Address</h5>
                                </div>

                                <div class="col-md-12">
                                    <div class="form-group">
                                        <label for="shipping_address">Address <span class="text-danger">*</span></label>
                                        <textarea name="shipping_address" id="shipping_address" class="form-control" rows="3" placeholder="Enter complete address" required></textarea>
                                    </div>
                                </div>

                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label for="city">City <span class="text-danger">*</span></label>
                                        <input type="text" name="city" id="city" class="form-control" placeholder="Enter city" required>
                                    </div>
                                </div>

                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label for="postal_code">Postal Code</label>
                                        <input type="text" name="postal_code" id="postal_code" class="form-control" placeholder="Enter postal code">
                                    </div>
                                </div>

                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label for="country">Country <span class="text-danger">*</span></label>
                                        <input type="text" name="country" id="country" class="form-control" placeholder="Enter country" required>
                                    </div>
                                </div>
                            </div>

                            <!-- Product Selection -->
                            <div class="row mb-4">
                                <div class="col-md-12">
                                    <h5 class="border-bottom pb-2 mb-3">Order Items</h5>
                                </div>

                                <div class="col-md-12">
                                    <div class="table-responsive">
                                        <table class="table table-bordered" id="order-items-table">
                                            <thead class="bg-light">
                                            <tr>
                                                <th width="40%">Product</th>
                                                <th width="15%">Unit Price</th>
                                                <th width="15%">Quantity</th>
                                                <th width="15%">Tax (%)</th>
                                                <th width="10%">Subtotal</th>
                                                <th width="5%">Action</th>
                                            </tr>
                                            </thead>
                                            <tbody id="order-items-body">
                                            <tr class="order-item-row">
                                                <td>
                                                    <select name="products[0][product_id]" class="form-control product-select" required>
                                                        <option value="">Select Product</option>
                                                        @foreach($products as $product)
                                                            <option value="{{ $product->id }}"
                                                                    data-price="{{ $product->unit_price }}"
                                                                    data-stock="{{ $product->current_stock }}"
                                                                    data-tax="{{ $product->tax ?? 0 }}">
                                                                {{ $product->name }} (Stock: {{ $product->current_stock }})
                                                            </option>
                                                        @endforeach
                                                    </select>
                                                </td>
                                                <td>
                                                    <input type="number" name="products[0][unit_price]" class="form-control unit-price" step="0.01" readonly>
                                                </td>
                                                <td>
                                                    <input type="number" name="products[0][quantity]" class="form-control quantity" min="1" value="1" required>
                                                </td>
                                                <td>
                                                    <input type="number" name="products[0][tax]" class="form-control tax" step="0.01" value="0">
                                                </td>
                                                <td>
                                                    <input type="text" class="form-control subtotal" readonly value="0.00">
                                                </td>
                                                <td>
                                                    <button type="button" class="btn btn-danger btn-sm remove-item" disabled>
                                                        <i class="fa fa-trash"></i>
                                                    </button>
                                                </td>
                                            </tr>
                                            </tbody>
                                        </table>
                                    </div>

                                    <button type="button" class="btn btn-primary btn-sm" id="add-item">
                                        <i class="fa fa-plus"></i> Add Product
                                    </button>
                                </div>
                            </div>

                            <!-- Order Summary -->
                            <div class="row">
                                <div class="col-md-7">
                                    <div class="form-group">
                                        <label for="order_note">Order Note</label>
                                        <textarea name="order_note" id="order_note" class="form-control" rows="3" placeholder="Add any special instructions..."></textarea>
                                    </div>
                                </div>

                                <div class="col-md-5">
                                    <div class="card">
                                        <div class="card-body">
                                            <h5 class="mb-3">Order Summary</h5>

                                            <div class="d-flex justify-content-between mb-2">
                                                <span>Subtotal:</span>
                                                <span id="order-subtotal">0.00</span>
                                            </div>

                                            <div class="d-flex justify-content-between mb-2">
                                                <span>Tax:</span>
                                                <span id="order-tax">0.00</span>
                                            </div>

                                            <div class="form-group mb-2">
                                                <label for="shipping_cost">Shipping Cost:</label>
                                                <input type="number" name="shipping_cost" id="shipping_cost" class="form-control" step="0.01" value="0">
                                            </div>

                                            <div class="form-group mb-2">
                                                <label for="discount">Discount:</label>
                                                <input type="number" name="discount" id="discount" class="form-control" step="0.01" value="0">
                                            </div>

                                            <hr>

                                            <div class="d-flex justify-content-between mb-3">
                                                <strong>Total:</strong>
                                                <strong id="order-total">0.00</strong>
                                            </div>

                                            <div class="form-group">
                                                <label for="payment_method">Payment Method <span class="text-danger">*</span></label>
                                                <select name="payment_method" id="payment_method" class="form-control" required>
                                                    <option value="cash_on_delivery">Cash on Delivery</option>
                                                    <option value="bank_transfer">Bank Transfer</option>
                                                    <option value="card">Card Payment</option>
                                                    <option value="mobile_banking">Mobile Banking</option>
                                                </select>
                                            </div>

                                            <div class="form-group">
                                                <label for="payment_status">Payment Status <span class="text-danger">*</span></label>
                                                <select name="payment_status" id="payment_status" class="form-control" required>
                                                    <option value="unpaid">Unpaid</option>
                                                    <option value="paid">Paid</option>
                                                </select>
                                            </div>

                                            <div class="form-group">
                                                <label for="delivery_status">Delivery Status <span class="text-danger">*</span></label>
                                                <select name="delivery_status" id="delivery_status" class="form-control" required>
                                                    <option value="pending">Pending</option>
                                                    <option value="confirmed">Confirmed</option>
                                                    <option value="processing">Processing</option>
                                                    <option value="shipped">Shipped</option>
                                                    <option value="delivered">Delivered</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Submit Buttons -->
                            <div class="row mt-4">
                                <div class="col-md-12">
                                    <button type="submit" class="btn btn-success">
                                        <i class="fa fa-check"></i> Create Order
                                    </button>
                                    <a href="#" class="btn btn-secondary">
                                        <i class="fa fa-times"></i> Cancel
                                    </a>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@section('script')
    <script>
        (function() {
            'use strict';

            // Wait for DOM to be ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', initOrderForm);
            } else {
                initOrderForm();
            }

            function initOrderForm() {
                let itemIndex = 1;

                // Try to initialize Select2 if available
                function initSelect2() {
                    if (typeof $.fn.select2 !== 'undefined') {
                        $('.product-select, #customer_id').each(function() {
                            if (!$(this).hasClass('select2-hidden-accessible')) {
                                $(this).select2({
                                    width: '100%',
                                    placeholder: 'Select an option'
                                });
                            }
                        });
                    }
                }

                // Initial Select2 attempt
                initSelect2();

                // Add new product row
                $('#add-item').on('click', function() {
                    const newRow = `
                <tr class="order-item-row">
                    <td>
                        <select name="products[${itemIndex}][product_id]" class="form-control product-select" required>
                            <option value="">Select Product</option>
                            @foreach($products as $product)
                    <option value="{{ $product->id }}"
                                    data-price="{{ $product->unit_price }}"
                                    data-stock="{{ $product->current_stock }}"
                                    data-tax="{{ $product->tax ?? 0 }}">
                                    {{ $product->name }} (Stock: {{ $product->current_stock }})
                                </option>
                            @endforeach
                    </select>
                </td>
                <td>
                    <input type="number" name="products[${itemIndex}][unit_price]" class="form-control unit-price" step="0.01" readonly>
                    </td>
                    <td>
                        <input type="number" name="products[${itemIndex}][quantity]" class="form-control quantity" min="1" value="1" required>
                    </td>
                    <td>
                        <input type="number" name="products[${itemIndex}][tax]" class="form-control tax" step="0.01" value="0">
                    </td>
                    <td>
                        <input type="text" class="form-control subtotal" readonly value="0.00">
                    </td>
                    <td>
                        <button type="button" class="btn btn-danger btn-sm remove-item">
                            <i class="fa fa-trash"></i>
                        </button>
                    </td>
                </tr>
            `;

                    $('#order-items-body').append(newRow);
                    initSelect2();
                    itemIndex++;
                    updateRemoveButtons();
                });

                // Remove product row
                $(document).on('click', '.remove-item', function() {
                    $(this).closest('tr').remove();
                    updateRemoveButtons();
                    calculateTotal();
                });

                // Update remove buttons state
                function updateRemoveButtons() {
                    const rowCount = $('.order-item-row').length;
                    $('.remove-item').prop('disabled', rowCount === 1);
                }

                // Handle product selection
                $(document).on('change', '.product-select', function() {
                    const row = $(this).closest('tr');
                    const selectedOption = $(this).find('option:selected');
                    const price = selectedOption.data('price') || 0;
                    const tax = selectedOption.data('tax') || 0;
                    const stock = selectedOption.data('stock') || 0;

                    row.find('.unit-price').val(parseFloat(price).toFixed(2));
                    row.find('.tax').val(parseFloat(tax).toFixed(2));
                    row.find('.quantity').attr('max', stock);

                    calculateRowSubtotal(row);
                });

                // Calculate row subtotal
                $(document).on('input', '.quantity, .unit-price, .tax', function() {
                    const row = $(this).closest('tr');
                    calculateRowSubtotal(row);
                });

                function calculateRowSubtotal(row) {
                    const quantity = parseFloat(row.find('.quantity').val()) || 0;
                    const unitPrice = parseFloat(row.find('.unit-price').val()) || 0;
                    const tax = parseFloat(row.find('.tax').val()) || 0;

                    const subtotal = quantity * unitPrice;
                    const taxAmount = (subtotal * tax) / 100;
                    const total = subtotal + taxAmount;

                    row.find('.subtotal').val(total.toFixed(2));
                    calculateTotal();
                }

                // Calculate order total
                $(document).on('input', '#shipping_cost, #discount', function() {
                    calculateTotal();
                });

                function calculateTotal() {
                    let subtotal = 0;
                    let totalTax = 0;

                    $('.order-item-row').each(function() {
                        const quantity = parseFloat($(this).find('.quantity').val()) || 0;
                        const unitPrice = parseFloat($(this).find('.unit-price').val()) || 0;
                        const tax = parseFloat($(this).find('.tax').val()) || 0;

                        const itemSubtotal = quantity * unitPrice;
                        const itemTax = (itemSubtotal * tax) / 100;

                        subtotal += itemSubtotal;
                        totalTax += itemTax;
                    });

                    const shippingCost = parseFloat($('#shipping_cost').val()) || 0;
                    const discount = parseFloat($('#discount').val()) || 0;
                    const grandTotal = subtotal + totalTax + shippingCost - discount;

                    $('#order-subtotal').text(subtotal.toFixed(2));
                    $('#order-tax').text(totalTax.toFixed(2));
                    $('#order-total').text(grandTotal.toFixed(2));
                }

                // Form validation
                $('#manual-order-form').on('submit', function(e) {
                    const hasProducts = $('.product-select').filter(function() {
                        return $(this).val() !== '';
                    }).length > 0;

                    if (!hasProducts) {
                        e.preventDefault();
                        alert('Please select at least one product');
                        return false;
                    }
                });
            }
        })();
    </script>
@endsection
