<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\PathaoToken;
use Carbon\Carbon;

class PathaoService
{
    protected $baseUrl;
    protected $clientId;
    protected $clientSecret;
    protected $username;
    protected $password;

    public function __construct()
    {
        $this->baseUrl = config('pathao.base_url');
        $this->clientId = config('pathao.client_id');
        $this->clientSecret = config('pathao.client_secret');
        $this->username = config('pathao.username');
        $this->password = config('pathao.password');
    }

    /**
     * Get valid access token (from DB or generate new)
     */
    public function getAccessToken($forceNew = false)
    {
        // If force new, skip cache check
        if (!$forceNew) {
            $token = PathaoToken::latest()->first();

            // Use existing token if still valid (with 10 min buffer for safety)
            if ($token && Carbon::now()->addMinutes(10)->lt($token->expires_at)) {
                Log::info('Using cached Pathao token', [
                    'expires_at' => $token->expires_at,
                    'time_remaining' => Carbon::now()->diffInMinutes($token->expires_at) . ' minutes'
                ]);
                return $token->access_token;
            }
        }

        // Generate new token
        return $this->issueToken();
    }

    /**
     * Issue new access token
     */
    private function issueToken()
    {
        try {
            Log::info('Generating new Pathao token', [
                'base_url' => $this->baseUrl,
                'client_id' => $this->clientId,
                'username' => $this->username
            ]);

            $response = Http::timeout(30)
                ->withHeaders([
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ])
                ->post($this->baseUrl . '/aladdin/api/v1/issue-token', [
                    'client_id' => $this->clientId,
                    'client_secret' => $this->clientSecret,
                    'grant_type' => 'password',
                    'username' => $this->username,
                    'password' => $this->password,
                ]);

            $data = $response->json();

            Log::info('Pathao token response', [
                'status' => $response->status(),
                'has_access_token' => isset($data['access_token']),
                'response_keys' => array_keys($data ?? [])
            ]);

            if (!isset($data['access_token'])) {
                Log::error('Pathao Token Generation Failed', [
                    'status' => $response->status(),
                    'response' => $data
                ]);
                throw new \Exception('Failed to generate Pathao access token: ' . ($data['message'] ?? 'Unknown error'));
            }

            // Clear old tokens and save new one
            PathaoToken::query()->delete();
            
            $tokenRecord = PathaoToken::create([
                'access_token' => $data['access_token'],
                'refresh_token' => $data['refresh_token'] ?? null,
                'expires_at' => now()->addSeconds($data['expires_in']),
            ]);

            Log::info('Pathao token saved successfully', [
                'expires_at' => $tokenRecord->expires_at,
                'expires_in_minutes' => round($data['expires_in'] / 60, 2)
            ]);

            return $data['access_token'];
            
        } catch (\Exception $e) {
            Log::error('Pathao Token Exception', [
                'message' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]);
            throw $e;
        }
    }

    /**
     * Make authenticated request with automatic token retry
     */
private function makeAuthenticatedRequest($method, $endpoint, $data = null, $retry = true)
{
    $token = $this->getAccessToken();

    $request = Http::timeout(30)
        ->withHeaders([
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $token,
        ]);

    $response = $method === 'GET'
        ? $request->get($this->baseUrl . $endpoint)
        : $request->post($this->baseUrl . $endpoint, $data);

    // Retry once if 401 Unauthorized
    if ($response->status() === 401 && $retry) {
        Log::warning('Token invalid, retrying with new token', [
            'endpoint' => $endpoint,
        ]);

        $newToken = $this->getAccessToken(true); // force new token

        $request = Http::timeout(30)
            ->withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $newToken,
            ]);

        $response = $method === 'GET'
            ? $request->get($this->baseUrl . $endpoint)
            : $request->post($this->baseUrl . $endpoint, $data);
    }

    return $response->json();
}




    /**
     * Get list of stores
     */
    public function getStores()
    {
        return $this->makeAuthenticatedRequest('GET', '/aladdin/api/v1/stores');
    }

    /**
     * Get list of cities
     */
    public function getCities()
    {
        return $this->makeAuthenticatedRequest('GET', '/aladdin/api/v1/city-list');
    }

    /**
     * Get zones for a city
     */
    public function getZones($cityId)
    {
        return $this->makeAuthenticatedRequest('GET', "/aladdin/api/v1/cities/{$cityId}/zone-list");
    }

    /**
     * Get areas for a zone
     */
    public function getAreas($zoneId)
    {
        return $this->makeAuthenticatedRequest('GET', "/aladdin/api/v1/zones/{$zoneId}/area-list");
    }

    /**
     * Calculate delivery price
     */
    public function calculatePrice(array $data)
    {
        return $this->makeAuthenticatedRequest('POST', '/aladdin/api/v1/merchant/price-plan', $data);
    }

    /**
     * Create a new order
     */
    public function createOrder(array $data)
    {
        Log::info('Creating Pathao Order', ['order_data' => $data]);
        $result = $this->makeAuthenticatedRequest('POST', '/aladdin/api/v1/orders', $data);
        Log::info('Pathao Order Result', ['result' => $result]);
        return $result;
    }

    /**
     * Create bulk orders
     */
    public function createBulkOrders(array $orders)
    {
        return $this->makeAuthenticatedRequest('POST', '/aladdin/api/v1/orders/bulk', ['orders' => $orders]);
    }

    /**
     * Get order info
     */
    public function getOrderInfo($consignmentId)
    {
        return $this->makeAuthenticatedRequest('GET', "/aladdin/api/v1/orders/{$consignmentId}/info");
    }

    /**
     * Create a new store
     */
    public function createStore(array $data)
    {
        return $this->makeAuthenticatedRequest('POST', '/aladdin/api/v1/stores', $data);
    }
}