<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\PathaoService;
use App\Models\Order;
use Illuminate\Support\Facades\Log;

class PathaoOrderController extends Controller
{
    protected $pathao;

    public function __construct(PathaoService $pathao)
    {
        $this->pathao = $pathao;
    }

    /**
     * Send order to Pathao
     */
    public function sendToPathao(Order $order)
    {
        try {
            // Check if already sent
            if ($order->pathao_consignment_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Order already sent to Pathao'
                ]);
            }

            // Get store ID from config
            $storeId = config('pathao.store_id');

            if (!$storeId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pathao store ID not configured. Please set PATHAO_STORE_ID in .env file'
                ]);
            }

            // Prepare shipping address
            $shippingAddress = $order->shipping_address;
            if (!$shippingAddress) {
                return response()->json([
                    'success' => false,
                    'message' => 'Shipping address not found for this order'
                ]);
            }

            // Calculate total weight (you can customize this)
            $totalWeight = 0.5; // Default 0.5 kg
            if (count($order->orderDetails) > 1) {
                $totalWeight = count($order->orderDetails) * 0.5;
            }

            // Prepare order data
            $orderData = [
                'store_id' => (int) $storeId,
                'merchant_order_id' => $order->code,
                'recipient_name' => 'mohsin',
                'recipient_phone' => '01715486265',
                'recipient_address' => 'dhaka, gopalgonj, bangladesh',
                'delivery_type' => config('pathao.delivery_types.normal'), // 48
                'item_type' => config('pathao.item_types.parcel'), // 2
                'item_quantity' => count($order->orderDetails),
                'item_weight' => $totalWeight,
                'amount_to_collect' => $order->payment_type === 'cash_on_delivery' ? (int) $order->grand_total : 0,
                'item_description' => 'Order #' . $order->code . ' from ' . config('app.name'),
            ];

            // Optional: Add special instructions
            if ($order->additional_info) {
                $orderData['special_instruction'] = $order->additional_info;
            }

            Log::info('Sending order to Pathao', $orderData);

            // Send to Pathao
            $response = $this->pathao->createOrder($orderData);

            // Check response
            if (!isset($response['data']['consignment_id'])) {
                Log::error('Pathao Order Failed', $response);
                
                $errorMessage = 'Failed to create Pathao order';
                if (isset($response['errors'])) {
                    $errors = [];
                    foreach ($response['errors'] as $key => $messages) {
                        $errors[] = implode(', ', $messages);
                    }
                    $errorMessage .= ': ' . implode('; ', $errors);
                } elseif (isset($response['message'])) {
                    $errorMessage .= ': ' . $response['message'];
                }

                return response()->json([
                    'success' => false,
                    'message' => $errorMessage,
                    'pathao_response' => $response
                ]);
            }

            // Update order with Pathao info
            $order->update([
                'pathao_consignment_id' => $response['data']['consignment_id'],
                'pathao_status' => $response['data']['order_status'] ?? 'Pending',
                'pathao_delivery_fee' => $response['data']['delivery_fee'] ?? null,
                'pathao_sent_at' => now(),
            ]);

            Log::info('Order sent to Pathao successfully', [
                'order_id' => $order->id,
                'consignment_id' => $response['data']['consignment_id']
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Order sent to Pathao successfully',
                'data' => $response['data']
            ]);

        } catch (\Exception $e) {
            Log::error('Pathao Order Exception: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get order status from Pathao
     */
    public function getOrderStatus(Order $order)
    {
        if (!$order->pathao_consignment_id) {
            return response()->json([
                'success' => false,
                'message' => 'Order not sent to Pathao yet'
            ]);
        }

        $response = $this->pathao->getOrderInfo($order->pathao_consignment_id);

        if (isset($response['data'])) {
            // Update order status
            $order->update([
                'pathao_status' => $response['data']['order_status'] ?? $order->pathao_status
            ]);

            return response()->json([
                'success' => true,
                'data' => $response['data']
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Failed to get order status',
            'response' => $response
        ]);
    }

    /**
     * Format phone number to BD format (11 digits)
     */
    private function formatPhoneNumber($phone)
    {
        // Remove all non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);

        // If starts with +880, remove it
        if (substr($phone, 0, 3) === '880') {
            $phone = '0' . substr($phone, 3);
        }

        // If starts with 880, remove it
        if (substr($phone, 0, 2) === '88') {
            $phone = substr($phone, 2);
        }

        // Ensure it starts with 0
        if (substr($phone, 0, 1) !== '0') {
            $phone = '0' . $phone;
        }

        // Ensure 11 digits
        $phone = substr($phone, 0, 11);

        // If less than 11 digits, use default
        if (strlen($phone) < 11) {
            return '01700000000';
        }

        return $phone;
    }

    /**
     * Format complete address
     */
    private function formatAddress($shippingAddress)
    {
        $parts = [];

        if ($shippingAddress->address) {
            $parts[] = $shippingAddress->address;
        }

        if ($shippingAddress->city) {
            $parts[] = $shippingAddress->city->name ?? $shippingAddress->city;
        }

        if ($shippingAddress->state) {
            $parts[] = $shippingAddress->state->name ?? $shippingAddress->state;
        }

        if ($shippingAddress->country) {
            $parts[] = $shippingAddress->country->name ?? $shippingAddress->country;
        }

        if ($shippingAddress->postal_code) {
            $parts[] = 'Postal Code: ' . $shippingAddress->postal_code;
        }

        $address = implode(', ', array_filter($parts));

        // Ensure minimum length (10 chars required)
        if (strlen($address) < 10) {
            $address = 'Address: ' . $address . ', Bangladesh';
        }

        // Ensure maximum length (220 chars max)
        if (strlen($address) > 220) {
            $address = substr($address, 0, 217) . '...';
        }

        return $address;
    }
}